#ifndef _TIMER_H
#define _TIMER_H

//------------------------------------------------CAPTURE-----------------------------------------------------------------------------
/** Captures que posee el LPC2114 \n
 * CAPnm_P0xx, donde: \n
 * n: nmero de timer, '0' para el TIMER0 o '1' para el TIMER1. \n
 * m: nmero de capture (0-3). \n
 * xx: nmero de pin del PORT0. \n
*/
enum CAPTURES{ CAP00_P02, CAP00_P022, CAP00_P030,       //CAP0.0
       CAP01_P04, CAP01_P027,                           //CAP0.1
       CAP02_P06, CAP02_P016, CAP02_P028,               //CAP0.2
       CAP03_P029,                                      //CAP0.3
       CAP10_P010,                                      //CAP0.0
       CAP11_P011,                                      //CAP0.1
       CAP12_P017, CAP12_P019,                          //CAP0.2
       CAP13_P018, CAP13_P021 };                        //CAP0.3

#define RISING_EDGE   0x01                      /**<Flanco de subida.*/
#define FALLING_EDGE  0x02                      /**<Flanco de bajada.*/
#define CAP_EVENT     0x04                      /**<Interrupcion tras un flanco de capture.*/

#define CAPTURE0 (1<<4)                         /**<Uso interna para macros.*/
#define CAPTURE1 (1<<5)                         /**<Uso interna para macros.*/
#define CAPTURE2 (1<<6)                         /**<Uso interna para macros.*/
#define CAPTURE3 (1<<7)                         /**<Uso interna para macros.*/

#define CAPTURE00_EVEN()  (T0IR & CAPTURE0)       /**<Chequea la bandera de interrupcion del Capture0.0*/
#define CAPTURE01_EVEN()  (T0IR & CAPTURE1)       /**<Chequea la bandera de interrupcion del Capture0.1*/
#define CAPTURE02_EVEN()  (T0IR & CAPTURE2)       /**<Chequea la bandera de interrupcion del Capture0.2*/
#define CAPTURE03_EVEN()  (T0IR & CAPTURE3)       /**<Chequea la bandera de interrupcion del Capture0.3*/
#define CAPTURE10_EVEN()  (T1_IR & CAPTURE0)      /**<Chequea la bandera de interrupcion del Capture1.0*/
#define CAPTURE11_EVEN()  (T1_IR & CAPTURE1)      /**<Chequea la bandera de interrupcion del Capture1.1*/
#define CAPTURE12_EVEN()  (T1_IR & CAPTURE2)      /**<Chequea la bandera de interrupcion del Capture1.2*/
#define CAPTURE13_EVEN()  (T1_IR & CAPTURE3)      /**<Chequea la bandera de interrupcion del Capture1.3*/

#define CAPTURE00_CLRFLAG()  (T0IR = CAPTURE0)    /**<Baja la bandera del Capture0.0*/
#define CAPTURE01_CLRFLAG()  (T0IR = CAPTURE1)    /**<Baja la bandera del Capture0.1*/
#define CAPTURE02_CLRFLAG()  (T0IR = CAPTURE2)    /**<Baja la bandera del Capture0.2*/
#define CAPTURE03_CLRFLAG()  (T0IR = CAPTURE3)    /**<Baja la bandera del Capture0.3*/
#define CAPTURE10_CLRFLAG()  (T1_IR = CAPTURE0)   /**<Baja la bandera del Capture1.0*/
#define CAPTURE11_CLRFLAG()  (T1_IR = CAPTURE1)   /**<Baja la bandera del Capture1.1*/
#define CAPTURE12_CLRFLAG()  (T1_IR = CAPTURE2)   /**<Baja la bandera del Capture1.2*/
#define CAPTURE13_CLRFLAG()  (T1_IR = CAPTURE3)   /**<Baja la bandera del Capture1.3*/

/** \fn void capture_init(unsigned char cap_pin, unsigned char mode, unsigned char priority)
 * \brief Esta funcin inicializa el modulo capture.
 * \param 1) "unsigned char cap_pin", es el capture con el que desea trabajar.
 * \param 2) "unsigned char mode", configura el modo en que ocurrir la captura del Timer Counter.
 * \param 3) "unsigned char priority", la prioridad de la interupcin por capture, si se utiliza
 * CAP_EVENT.
 * El modo puede ser por Flanco de Subida, Flanco de Bajada y si se genera Interrupcin. Se forma a
 * partir de los defines RISING_EDGE, FALLING_EDGE y CAP_EVENT. Por ejemplo (RISING_EDGE | CAP_EVENT).
 * \return   No devuelve ningn valor.
 * \author   Lucas Martini
 */
void capture_init(unsigned char cap_pin, unsigned char mode, unsigned char priority);
/** \fn void capture_reset(unsigned char cap_pin)
 * \brief Esta funcin desactiva el modulo capture, configurando el pin como GPIO.
 * \param 1) "unsigned char cap_pin", es el capture con el que desea desactivar.
 * \return   No devuelve ningn valor.
 * \author   Lucas Martini
 */
void capture_reset(unsigned char cap_pin);
/** \fn void capture_get(unsigned char cap_pin)
 * \brief Esta funcion devuelve el valor de la captura.
 * \param 1) "unsigned char cap_pin", es el capture que se desea sensar.
 * \return   Devuelve el valor de la captura (valor en pclk transcurridos entre eventos)
 * \author   Lucas Martini
 */
int capture_get(unsigned char cap_pin);
//------------------------------------------------TIMER-----------------------------------------------------------------------------
#ifdef TIMER_C
#define EXTERN extern
#else
#define EXTERN 
#endif

#define TIMER0  0                         /**<Timer 0.*/
#define TIMER1  1                         /**<Timer 1.*/
//Tareas a realizar tras un match
#define INTERRUPT_ON_MATCH  0x01          /**<Interrupmpir tras un match.*/
#define RESET_ON_MATCH      0x02          /**<Resetear contador tras un match.*/
#define STOP_ON_MATCH       0x04          /**<Detener el contador tras un match.*/

/** 
 * \def TIMER0_ENABLE()
 * \brief Macro que habilita Timer Counter y Preescaler Counter (TIMER0).
 */
#define TIMER0_ENABLE()   T0TCR = 0x1     
/** 
 * \def TIMER0_DISABLE()
 * \brief Macro que deshabilita Timer Counter y Preescaler Counter (TIMER0).
 */
#define TIMER0_DISABLE()  T0TCR = 0x0     
/** 
 * \def TIMER0_RESET()
 * \brief Macro que resetea el TIMER0.
 */
#define TIMER0_RESET()    T0TCR = 0x2; T0TCR &= 0x1
/** 
 * \def TIMER1_ENABLE()
 * \brief Macro que habilita Timer Counter y Preescaler Counter (TIMER1).
 */
#define TIMER1_ENABLE()   T1_TCR = 0x1
/** 
 * \def TIMER1_DISABLE()
 * \brief Macro que deshabilita Timer Counter y Preescaler Counter (TIMER1).
 */
#define TIMER1_DISABLE()  T1_TCR = 0x0
/** 
 * \def TIMER1_RESET()
 * \brief Macro que resetea el TIMER1.
 */
#define TIMER1_RESET()    T1_TCR = 0x2; T1_TCR &= 0x1

#define TIMER_COUNTER       0x0    /**<Se utiliza con la funcin timer_get() para obtener el Time Counter.*/   
#define PRESCALER           0x1    /**<Se utiliza con la funcin timer_get() para obtener el Prescaler.*/
#define PRESCALER_COUNTER   0x2    /**<Se utiliza con la funcin timer_get() para obtener el Prescaler Counter.*/


#define TIMER_MATCH0 0  /**<Se utiliza con la mayora de las funciones para indicar el MATCH0.*/
#define TIMER_MATCH1 1  /**<Se utiliza con la mayora de las funciones para indicar el MATCH0.*/
#define TIMER_MATCH2 2  /**<Se utiliza con la mayora de las funciones para indicar el MATCH0.*/
#define TIMER_MATCH3 3  /**<Se utiliza con la mayora de las funciones para indicar el MATCH0.*/

#define TIMER_M0 1
#define TIMER_M1 2
#define TIMER_M2 4
#define TIMER_M3 8

/** 
 * \def INTERRUPT_T0MR0()
 * \brief Macro que devuelve la bandera de la interrupcion del MATCH0 del TIMER0.
 */
#define INTERRUPT_T0MR0()  (T0IR & TIMER_M0) 
/** 
 * \def INTERRUPT_T0MR1()
 * \brief Macro que devuelve la bandera de la interrupcion del MATCH1 del TIMER0.
 */
#define INTERRUPT_T0MR1()  (T0IR & TIMER_M1)   /**<Bandera de interrupcion del MATCH1 del TIMER0.*/
/** 
 * \def INTERRUPT_T0MR2()
 * \brief Macro que devuelve la bandera de la interrupcion del MATCH2 del TIMER0.
 */
#define INTERRUPT_T0MR2()  (T0IR & TIMER_M2)   
/** 
 * \def INTERRUPT_T0MR3()
 * \brief Macro que devuelve la bandera de la interrupcion del MATCH3 del TIMER0.
 */
#define INTERRUPT_T0MR3()  (T0IR & TIMER_M3)
/** 
 * \def INTERRUPT_T1MR0()
 * \brief Macro que devuelve la bandera de la interrupcion del MATCH0 del TIMER1.
 */
#define INTERRUPT_T1MR0()  (T1_IR & TIMER_M0)
/** 
 * \def INTERRUPT_T1MR1()
 * \brief Macro que devuelve la bandera de la interrupcion del MATCH1 del TIMER1.
 */
#define INTERRUPT_T1MR1()  (T1_IR & TIMER_M1)
/** 
 * \def INTERRUPT_T1MR2()
 * \brief Macro que devuelve la bandera de la interrupcion del MATCH2 del TIMER1.
 */
#define INTERRUPT_T1MR2()  (T1_IR & TIMER_M2)
/** 
 * \def INTERRUPT_T1MR3()
 * \brief Macro que devuelve la bandera de la interrupcion del MATCH3 del TIMER1.
 */
#define INTERRUPT_T1MR3()  (T1_IR & TIMER_M3)

/** 
 * \def T0MR0_CLRFLAG() 
 * \brief Macro que baja la bandera del MATCH0 del TIMER0.
 */
#define T0MR0_CLRFLAG()  (T0IR = TIMER_M0)
/** 
 * \def T0MR1_CLRFLAG() 
 * \brief Macro que baja la bandera del MATCH1 del TIMER0.
 */
#define T0MR1_CLRFLAG()  (T0IR = TIMER_M1)
/** 
 * \def T0MR2_CLRFLAG() 
 * \brief Macro que baja la bandera del MATCH2 del TIMER0.
 */
#define T0MR2_CLRFLAG()  (T0IR = TIMER_M2)
/** 
 * \def T0MR3_CLRFLAG() 
 * \brief Macro que baja la bandera del MATCH3 del TIMER0.
 */
#define T0MR3_CLRFLAG()  (T0IR = TIMER_M3)
/** 
 * \def T1MR0_CLRFLAG() 
 * \brief Macro que baja la bandera del MATCH0 del TIMER1.
 */
#define T1MR0_CLRFLAG()  (T1_IR = TIMER_M0)
/** 
 * \def T1MR1_CLRFLAG() 
 * \brief Macro que baja la bandera del MATCH1 del TIMER1.
 */
#define T1MR1_CLRFLAG()  (T1_IR = TIMER_M1)
/** 
 * \def T1MR2_CLRFLAG() 
 * \brief Macro que baja la bandera del MATCH2 del TIMER1.
 */
#define T1MR2_CLRFLAG()  (T1_IR = TIMER_M2)
/** 
 * \def T1MR3_CLRFLAG() 
 * \brief Macro que baja la bandera del MATCH3 del TIMER1.
 */
#define T1MR3_CLRFLAG()  (T1_IR = TIMER_M3)

int timer_flag;   /**<Variable global utilizada en la funcin timer_mr0_mr1_enable(void) y en la interrupcin del timer1.*/
int T1_match0;    /**<Variable global utilizada en la funcin timer_mr0_mr1_enable(void) y en la interrupcin del timer1.*/

/** \fn irq_timer1 (void) __attribute__ ((interrupt,noinline)).
 * \brief Esta funcin atiende la interrupcion del timer 1.
 * \warning Queda a cargo del programador trabajar con los callbacks correspondientes.
 * \author Nstor Levi Palomeque
 */
void irq_timer1 (void) __attribute__ ((interrupt ("IRQ"),noinline));
/** \fn irq_timer0 (void) __attribute__ ((interrupt,noinline)).
 * \brief Esta funcin atiende la interrupcion del timer 0.
 * \warning Queda a cargo del programador trabajar con los callbacks correspondientes.
 * \author Nstor Levi Palomeque
 */
void irq_timer0 (void) __attribute__ ((interrupt ("IRQ"),noinline));

/** \fn EXTERN void (*callbacks_timer1_match0)(void);.
 * \brief Este puntero a una funcin es un callback que es llamado desda la interrupcion del timer1 cuando la
 * fuente de la misma es provocada por el MATCH0.
 * \warning Queda a cargo del programador trabajar codificar este callback.
 * \author Nstor Levi Palomeque
 */
EXTERN void (*callbacks_timer1_match0)(void);
/** \fn EXTERN void callbacks_timer1_match1(void);.
 * \brief Este puntero a una funcin es un callback que es llamado desda la interrupcion del timer1 cuando la
 * fuente de la misma es provocada por el MATCH1.
 * \warning Queda a cargo del programador trabajar codificar este callback.
 * \author Nstor Levi Palomeque
 */
EXTERN void (*callbacks_timer1_match1)(void);
/** \fn EXTERN void callbacks_timer1_match2(void);.
 * \brief Este puntero a una funcin es un callback que es llamado desda la interrupcion del timer1 cuando la
 * fuente de la misma es provocada por el MATCH2.
 * \warning Queda a cargo del programador trabajar codificar este callback.
 * \author Nstor Levi Palomeque
 */
EXTERN void (*callbacks_timer1_match2)(void);
/** \fn EXTERN void callbacks_timer1_match3(void);.
 * \brief Este puntero a una funcin es un callback que es llamado desda la interrupcion del timer1 cuando la
 * fuente de la misma es provocada por el MATCH3.
 * \warning Queda a cargo del programador trabajar codificar este callback.
 * \author Nstor Levi Palomeque
 */
EXTERN void (*callbacks_timer1_match3)(void);
/** \fn EXTERN void callbacks_timer1_capture0(void);.
 * \brief Este puntero a una funcin es un callback que es llamado desda la interrupcion del timer1 cuando la
 * fuente de la misma es provocada por el CAPTURE0.
 * \warning Queda a cargo del programador trabajar codificar este callback.
 * \author Nstor Levi Palomeque
 */
EXTERN void (*callbacks_timer1_capture0)(void);
/** \fn EXTERN void callbacks_timer1_capture1(void);.
 * \brief Este puntero a una funcin es un callback que es llamado desda la interrupcion del timer1 cuando la
 * fuente de la misma es provocada por el CAPTURE1.
 * \warning Queda a cargo del programador trabajar codificar este callback.
 * \author Nstor Levi Palomeque
 */
EXTERN void (*callbacks_timer1_capture1)(void);
/** \fn EXTERN void callbacks_timer1_capture2(void);.
 * \brief Este puntero a una funcin es un callback que es llamado desda la interrupcion del timer1 cuando la
 * fuente de la misma es provocada por el CAPTURE2.
 * \warning Queda a cargo del programador trabajar codificar este callback.
 * \author Nstor Levi Palomeque
 */
EXTERN void (*callbacks_timer1_capture2)(void);
/** \fn EXTERN void callbacks_timer1_capture3(void);.
 * \brief Este puntero a una funcin es un callback que es llamado desda la interrupcion del timer1 cuando la
 * fuente de la misma es provocada por el CAPTURE1.
 * \warning Queda a cargo del programador trabajar codificar este callback.
 * \author Nstor Levi Palomeque
 */
EXTERN void (*callbacks_timer1_capture3)(void);
/** \fn EXTERN void callbacks_timer0_match0(void);.
 * \brief Este puntero a una funcin es un callback que es llamado desda la interrupcion del timer0 cuando la
 * fuente de la misma es provocada por el MATCH0.
 * \warning Queda a cargo del programador trabajar codificar este callback.
 * \author Nstor Levi Palomeque
 */
EXTERN void (*callbacks_timer0_match0)(void);
/** \fn EXTERN void callbacks_timer0_match1(void);.
 * \brief Este puntero a una funcin es un callback que es llamado desda la interrupcion del timer0 cuando la
 * fuente de la misma es provocada por el MATCH1.
 * \warning Queda a cargo del programador trabajar codificar este callback.
 * \author Nstor Levi Palomeque
 */
EXTERN void (*callbacks_timer0_match1)(void);
/** \fn EXTERN void callbacks_timer0_match2(void);.
 * \brief Este puntero a una funcin es un callback que es llamado desda la interrupcion del timer0 cuando la
 * fuente de la misma es provocada por el MATCH2.
 * \warning Queda a cargo del programador trabajar codificar este callback.
 * \author Nstor Levi Palomeque
 */
EXTERN void (*callbacks_timer0_match2)(void);
/** \fn EXTERN void callbacks_timer0_match3(void);.
 * \brief Este puntero a una funcin es un callback que es llamado desda la interrupcion del timer0 cuando la
 * fuente de la misma es provocada por el MATCH3.
 * \warning Queda a cargo del programador trabajar codificar este callback.
 * \author Nstor Levi Palomeque
 */
EXTERN void (*callbacks_timer0_match3)(void);
/** \fn EXTERN void callbacks_timer0_capture0(void);.
 * \brief Este puntero a una funcin es un callback que es llamado desda la interrupcion del timer0 cuando la
 * fuente de la misma es provocada por el CAPTURE0.
 * \warning Queda a cargo del programador trabajar codificar este callback.
 * \author Nstor Levi Palomeque
 */
EXTERN void (*callbacks_timer0_capture0)(void);
/** \fn EXTERN void callbacks_timer0_capture1(void);.
 * \brief Este puntero a una funcin es un callback que es llamado desda la interrupcion del timer0 cuando la
 * fuente de la misma es provocada por el CAPTURE1.
 * \warning Queda a cargo del programador trabajar codificar este callback.
 * \author Nstor Levi Palomeque
 */
EXTERN void (*callbacks_timer0_capture1)(void);
/** \fn EXTERN void callbacks_timer0_capture2(void);.
 * \brief Este puntero a una funcin es un callback que es llamado desda la interrupcion del timer0 cuando la
 * fuente de la misma es provocada por el CAPTURE2.
 * \warning Queda a cargo del programador trabajar codificar este callback.
 * \author Nstor Levi Palomeque
 */
EXTERN void (*callbacks_timer0_capture2)(void);
/** \fn EXTERN void callbacks_timer0_capture3(void);.
 * \brief Este puntero a una funcin es un callback que es llamado desda la interrupcion del timer0 cuando la
 * fuente de la misma es provocada por el CAPTURE3.
 * \warning Queda a cargo del programador trabajar codificar este callback.
 * \author Nstor Levi Palomeque
 */
EXTERN void (*callbacks_timer0_capture3)(void);
/** \fn void timer_init(unsigned char timer, unsigned char match_n, char mode, unsigned char priority, int prescaler, int match).
 * \brief Esta funcin inicializa el timer.
 * \param 1) "timer" (TIMER0  TIMER1), indica cual es el timer que se inicializa.
 * \param 2) "match_n" (TIMER_MATCH0, TIMER_MATCH1, TIMER_MATCH2  TIMER_MATCH3) indica el registro MATCH donde se carga el valor a alcanzar.
 * \param 3) "mode" (INTERRUPT_ON_MATCH, RESET_ON_MATCH ó STOP_ON_MATCH) indica la accion a llevar a cabo cuando se alcanza el valor.
 * \param 4) "priority" es la prioridad de la interrupcin de dicho timer en caso de haberla
 * habilitado.
 * \param 5) "prescaler" se carga el valor que tendra el prescaler (0+1).
 * \param 6) "match" Se carga el valor a alzanzar.
 * \return   No retorna ningn valor.
 * \author   Lucas Martini
 */
void timer_init(unsigned char timer, unsigned char match_n, char mode, unsigned char priority, int prescaler, int match);
/** \fn void timer_reset(unsigned char timer, unsigned char match_n).
 * \brief Esta funcin resetea el modulo timer.
 * \param 1) "timer" (TIMER0  TIMER1), indica cual es el timer que se desabilita.
 * \param 2) "match_n" (TIMER_MATCH0, TIMER_MATCH1, TIMER_MATCH2  TIMER_MATCH3) indica el registro MATCH a resetear.
 * \return   No retorna ningn valor.
 * \warning Cada vez que se ejecuta esta funcion, por ejemplo para cambiar la cuenta de algun match,
 * se debe volver asignar todos los callbacks utilizados.
 * \author   Lucas Martini
 */
void timer_reset(unsigned char timer, unsigned char match_n);
/** \fn int timer_get(unsigned char timer, unsigned char contador).
 * \brief Esta funcin retorna informacin de los contadores del timer.
 * \param 1) "timer" (TIMER0  TIMER1), indica el timer al que se quiere pedir informacin.
 * \param 2) "contador" (TIMER_COUNTER, PRESCALER  PRESCALER_COUNTER) indica el contador a
 * retornar.
 * \return   Devuelve el valor del contador solicitado.
 * \author   Lucas Martini
 */
int timer_get(unsigned char timer, unsigned char contador);
/** \fn void timer_mr0_mr1_enable(void).
 * \brief Esta funcion se utiliza para emplear al MATCH0 y MATCH1 con dos bases de tiempo distintas. Por ej con MATCH0
 * se pueden generar interrupciones cada 400uSeg y con MATCH1 cada 1mSeg.
 * \warning Cabe aclarar que MATCH1 debe tener la mayor base de tiempo (en caso de no serlo esta funcion intercambia
 * su valor con el de MATCH0) pero debe tenerse en cuenta el callbacks que atiende a cada match. Por
 * ejemplo: si asignara al MATCH1 con 400uSeg y al MATCH0 con 1mSeg, esta funcin intercambia sus
 * valores (deja al MATCH1 con 1mSeg y al MATCH0 con 400uSeg) y el programador podra confundirse
 * con su asignacon inicial y programar el callback del MATCH0 esperando que el mismo tenga una
 * base de tiempo de 1mSeg siendo esto incorrecto.
 * Tambien debe configurarse para el MATCH1 el reset del time counter.
 * En caso de que las bases de tiempos esten muy cercanas (+- 5uSeg) se consideran como una sola interrupcin 
 * ejecutndose los dos callbacks (del MATCH0 y del MATCH1).
 * \return   No devuelve ningn valor.
 * \author   Nstor Levi Palomeque
 */
void timer_mr0_mr1_enable(void);
/** \fn void timer_mr0_mr1_disable(void).
 * \brief Esta funcin deshabilita los dos match 0 y 1. Resetea el modo ("mode") configurado con la
 * funcin timer_init(...,mode,...);
 * \warning  Deja configurada la interrupcin por timer1.
 * \return   No devuelve ningn valor.
 * \author   Nstor Levi Palomeque
 */
void timer_mr0_mr1_disable(void);
/** \fn void timer_disable_interrupt(unsigned char timer, unsigned char match_n).
 * \brief Esta funcin deshabilita la interrupcin del match correspondientemente indicado en "match_n" del timer dado 
 * por la variable "timer".
 * \return   No devuelve ningn valor.
 * \author   Nstor Levi Palomeque
 */
void timer_disable_interrupt(unsigned char timer, unsigned char match_n);
/** \fn void timer_enable_interrupt(unsigned char timer, unsigned char match_n).
 * \brief Esta funcin habilita la interrupcin del match correspondientemente indicado en "match_n" del timer dado por
 * la variable "timer".
 * \return   No devuelve ningn valor.
 * \author   Nstor Levi Palomeque
 */
void timer_enable_interrupt(unsigned char timer, unsigned char match_n);
/** \fn void callbacks_error().
 * \brief Esta se ejecuta cuando se intenta acceder a un callback que no ha sido implementado en alguna parte del programa.
 * \return   No devuelve ningn valor.
 * \author   Nstor Levi Palomeque
 */
void timer_callbacks_error();
#endif
