/*
 * @brief UART interrupt example with ring buffers
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2013
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */

#include "chip.h"
#include "board.h"
#include "sapi.h"
#include "sapi_uart.h"

#include "string.h"
#include "stdlib.h"

#include "comunicacion.h"



/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/
/* Transmit and receive ring buffers */
STATIC RINGBUFF_T txring, rxring;

/* Transmit and receive ring buffer sizes */
#define UART_SRB_SIZE 128	/* Send */
#define UART_RRB_SIZE 32	/* Receive */

/* Transmit and receive buffers */
static uint8_t rxbuff[UART_RRB_SIZE], txbuff[UART_SRB_SIZE];

const char inst1[] = "\r\nRobot EduRoMAA-CIAA\r\n";
const char inst2[] = "Con MiniCLI\r\n";


/*  CLI: Variables de Comandos */

char   CommandLine[COMMAND_BUFFER_LENGTH + 1];  //Read commands into this buffer from Serial.  +1 in length for a termination char

const char *delimiters            = ", \n";     //commands can be separated by return, space or comma

/*******************************
     your Command Names Here
*/
const char *addCommandToken       = "add";  	//Modify here
const char *subtractCommandToken  = "sub";      //Modify here
/*  CLI: Macros y Definiciones */

/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/

/*****************************************************************************
 * Private functions
 ****************************************************************************/

/*****************************************************************************
 * Public functions
 ****************************************************************************/
/**
 * @brief	UART interrupt handler using ring buffers
 * @return	Nothing
 */
void UARTx_IRQHandler(void)
{
	/* Want to handle any errors? Do it here. */

	/* Use default ring buffer handler. Override this with your own
	   code if you need more capability. */
	Chip_UART_IRQRBHandler(LPC_UARTX, &rxring, &txring);
}

int init_usb_com(void)
{
	/* sAPI.boardConfig(): Inicializa GPIO y configura USB, Teclas, LEDs, puerto I/O */
	boardConfig();

	uartConfig(UART_USB, 115200);
	//	Board_UART_Init(LPC_UARTX);

	/* Setup UART for 115.2K8N1 */
	Chip_UART_Init(LPC_UARTX);
	Chip_UART_SetBaud(LPC_UARTX, 115200);
	Chip_UART_ConfigData(LPC_UARTX, (UART_LCR_WLEN8 | UART_LCR_SBS_1BIT));
	Chip_UART_SetupFIFOS(LPC_UARTX, (UART_FCR_FIFO_EN | UART_FCR_TRG_LEV2));
	Chip_UART_TXEnable(LPC_UARTX);

	/* Before using the ring buffers, initialize them using the ring
	   buffer init function */
	RingBuffer_Init(&rxring, rxbuff, 1, UART_RRB_SIZE);
	RingBuffer_Init(&txring, txbuff, 1, UART_SRB_SIZE);

	/* Reset and enable FIFOs, FIFO trigger level 3 (14 chars) */
	Chip_UART_SetupFIFOS(LPC_UARTX, (UART_FCR_FIFO_EN | UART_FCR_RX_RS |
			UART_FCR_TX_RS | UART_FCR_TRG_LEV3));

	/* Enable receive data and line status interrupt */
	Chip_UART_IntEnable(LPC_UARTX, (UART_IER_RBRINT | UART_IER_RLSINT));

	/* preemption = 1, sub-priority = 1 */
	NVIC_SetPriority(UARTx_IRQn, 1);
	NVIC_EnableIRQ(UARTx_IRQn);

	/* Send initial messages */
	Chip_UART_SendRB(LPC_UARTX, &txring, inst1, sizeof(inst1) - 1);
	Chip_UART_SendRB(LPC_UARTX, &txring, inst2, sizeof(inst2) - 1);

	return 1;
}


int print(const char *cadena){
	uint8_t idx = 0;

	idx = Chip_UART_SendRB(LPC_UARTX, &txring, (const uint8_t *) cadena, strlen(cadena));

	return idx;
}

int println(char *cadena){
	uint8_t idx = 0;

	idx = Chip_UART_SendRB(LPC_UARTX, &txring, cadena, strlen(cadena));
	idx = Chip_UART_SendRB(LPC_UARTX, &txring, "\r\n", 2);

	return idx+2;
}

int print_int(int valor){
	uint8_t idx = 0;
	char cadena[10];

	itoa(valor, cadena, 10);
	idx = Chip_UART_SendRB(LPC_UARTX, &txring, cadena, strlen(cadena));

	return idx;
}

/* Transforma flotantes en cadena de 8 dgitos + 2 para "\r\n". */
int print_float8dig(float valor){
	uint8_t idx = 0;
	char cadena[10];
	char cad_aux[10];
	int entero;

	if(valor < 0){
		strcat(cadena, "-");
	}


	entero = (int) valor;
	/*Mete la parte entera a la cadena*/
	itoa(entero, cadena, 10);
	strcat(cadena, ".");


	while (strlen(cadena) < 8){

		valor -= (float) entero;
		valor *= (float) 10;

		entero = (int) valor;
		/*Mete la parte entera a la cadena*/
		itoa(abs(entero), cad_aux, 10);
		strcat(cadena,cad_aux);

	}

	strcat(cadena, "\r\n");
	idx = Chip_UART_SendRB(LPC_UARTX, &txring, cadena, strlen(cadena));

	return idx;
}


/* ****************************************************************************
 * Mquina de Comunicacin:
 * Llamada desde el lazo main. Hace ejecutar los comandos pendientes.
 */
int comunicate(void)
{
	bool received;

	received = getCommandLineFromSerialPort(CommandLine);
	if (received) {
		DoMyCommand(CommandLine);
	}
	return 1;
}


/*************************************************************************************************************
    getCommandLineFromSerialPort()
      Return the string of the next command. Commands are delimited by return"
      Handle BackSpace character
      Make all chars lowercase
 *************************************************************************************************************/

bool getCommandLineFromSerialPort(char * commandLine) {

	static uint8_t charsRead = 0;                      //note: COMAND_BUFFER_LENGTH must be less than 255 chars long
	//read asynchronously until full command input

	uint8_t key;
	uint8_t auxiliar[] = {BS, SPACE,BS};
	int bytes;

	/*	Si hay bytes en la cola de recepcin, los procesa. Si no, sigue de largo. */
	while ((bytes = Chip_UART_ReadRB(LPC_UARTX, &rxring, &key, 1)) > 0) {

		switch (key) {
		case CR:      //likely have full command in buffer now, commands are terminated by CR and/or LS
		case LF:
			commandLine[charsRead] = NULLCHAR;       //null terminate our command char array
			if (charsRead > 0)  {
				charsRead = 0;                           //charsRead is static, so have to reset
				return true;
			}
			break;
		case BS:                                    // handle backspace in input: put a space in last char
			if (charsRead > 0) {                        //and adjust commandLine and charsRead
				commandLine[--charsRead] = NULLCHAR;

				Chip_UART_SendRB(LPC_UARTX, &txring, auxiliar, 3);
			}
			break;
		default:
			if (charsRead < COMMAND_BUFFER_LENGTH) {
				commandLine[charsRead++] = key;
				Chip_UART_SendRB(LPC_UARTX, &txring, (const uint8_t *) &key, 1);
			}
			commandLine[charsRead] = NULLCHAR;     //just in case
			break;
		}
	}
	return false;
}


/* ****************************
   readNumber: return a 16bit (for Arduino Uno) signed integer from the command line
   readWord: get a text word from the command line

 */
int readNumber () {
	char * numTextPtr = strtok(NULL, delimiters);         //K&R string.h  pg. 250
	return atoi(numTextPtr);                              //K&R string.h  pg. 251
}

char * readWord() {
	char * word = strtok(NULL, delimiters);               //K&R string.h  pg. 250
	return word;
}

void nullCommand(char * ptrToCommandName) {
	print("> Comando invlido: ");
	println(ptrToCommandName);
}


/****************************************************
   Add your commands here
 */

int addCommand() {                                      //Modify here
	int firstOperand = readNumber();
	int secondOperand = readNumber();
	return firstOperand + secondOperand;
}

int subtractCommand() {                                //Modify here
	int firstOperand = readNumber();
	int secondOperand = readNumber();
	return firstOperand - secondOperand;
}

/****************************************************
   DoMyCommand
 */
bool DoMyCommand(char * commandLine) {
	//  print2("\nCommand: ", commandLine);
	int result;

	char * ptrToCommandName = strtok(commandLine, delimiters);
	//  print2("commandName= ", ptrToCommandName);

	if (strcmp(ptrToCommandName, addCommandToken) == 0) {                   //Modify here
		result = addCommand();
		print2("\r\n>    The sum is = ", result);


	} else {
		if (strcmp(ptrToCommandName, subtractCommandToken) == 0) {           //Modify here
			result = subtractCommand();                                       //K&R string.h  pg. 251
			print2("\r\n>    The difference is = ", result);
		} else {
			nullCommand(ptrToCommandName);
		}
	}
	return true;
}
