# -*- coding: utf-8 -*-
""" Comparison with paper "Pol-SAR Classification Based on Generalized Polar Decomposition of Mueller Matrix"

Usage:
  comparison_mueller.py --dataset=<str>

Options:
  --dataset=<str>    path to the dataset
  -h --help          show this screen.
"""

import sys
from os import path
from docopt import docopt
import numpy as np
import pandas

from sklearn import metrics

import vrl


sys.path.append('.')
import sartb


def comparison(ipath, dataset_name, unary_method, wishart_init, polsar_mode, ncomp, dof, ntimes,
               seed=1234567):
    rng = vrl.base.RNG(seed=seed)
    acc = []
    for i in range(ntimes):
        seed = rng.randint(0, 12345678)
        dataset = sartb.get_dataset(ipath, dataset_name, mode=polsar_mode, seed=seed)
        #dataset.overlap = False
        dataset.overlap = True

        #Using 2000 samples for each class as training samples
        y = sartb.segmentation_with_smoothing(dataset, unary_method=unary_method, wishart_init=wishart_init,
                                              ncomp=ncomp, seed=seed, dof=dof, nblocks=2000, block_size=1)
        confusion = metrics.confusion_matrix(dataset._label.ravel(), y.ravel(), range(dataset.nclasses))
        true = np.array(np.diagonal(confusion), dtype=np.float)
        all = np.array(np.sum(confusion, axis=1), dtype=np.float)
        acc_ = np.divide(true, all)
        acc.append(acc_)

    mean = np.mean(acc, axis=0)
    std = np.std(acc, axis=0)

    df = pandas.DataFrame([mean, std], columns=dataset._classes_names)
    print(df.to_string(index=False))

    return mean, std


if __name__ == "__main__":
    args = docopt(__doc__)
    ipath = path.abspath(str(args["--dataset"]))

    ntimes = 10
    dof = 5
    ncomp = 16

    dn = sartb.DatasetNames()

    wishart_init = 'kmeans'

    mean, std = comparison(ipath, dn.FlevolandMuellerMatrix, 'fisher_vector', wishart_init, 'real', ncomp,
                           dof, ntimes)
